import React, { useState } from 'react';
import Header from './components/Header';
import BottomNav from './components/BottomNav';
import HomePage from './components/HomePage';
import CreatePage from './components/CreatePage';
import LibraryPage from './components/LibraryPage';
import MorePage from './components/MorePage';
import MiniPlayer from './components/MiniPlayer';
import PlayerModal from './components/PlayerModal';
import { Song } from './types';

const mockSongs: Song[] = [
  { id: 1, rank: '01', title: 'Sunset Drive', artist: 'Cyberwave', artistAvatar: 'https://i.pravatar.cc/32?u=cyberwave', artistVerified: true, tags: 'Synthwave', imageUrl: 'https://picsum.photos/seed/s1/200' },
  { id: 2, rank: '02', title: 'Lost in the Echoes', artist: 'Nora Night', artistAvatar: 'https://i.pravatar.cc/32?u=noranight', artistVerified: false, tags: 'Chillstep', imageUrl: 'https://picsum.photos/seed/s2/200' },
  { id: 3, rank: '03', title: 'Neon Dreams', artist: 'Vector Shift', artistAvatar: 'https://i.pravatar.cc/32?u=vectorshift', artistVerified: true, tags: 'Retrowave', imageUrl: 'https://picsum.photos/seed/s3/200' },
  { id: 4, rank: '04', title: 'Quantum Leap', artist: 'Glitch Mob', artistAvatar: 'https://i.pravatar.cc/32?u=glitchmob', artistVerified: true, tags: 'IDM', imageUrl: 'https://picsum.photos/seed/s4/200' },
  { id: 5, rank: '05', title: 'Cosmic Dust', artist: 'Stellardrone', artistAvatar: 'https://i.pravatar.cc/32?u=stellardrone', artistVerified: false, tags: 'Ambient', imageUrl: 'https://picsum.photos/seed/s5/200' },
];

const App: React.FC = () => {
  const [activeTab, setActiveTab] = useState('home');
  const [currentSong, setCurrentSong] = useState<Song | null>(mockSongs[0]);
  const [isPlaying, setIsPlaying] = useState(false);
  const [isPlayerModalOpen, setPlayerModalOpen] = useState(false);

  const handlePlaySong = (song: Song) => {
    if (currentSong?.id === song.id) {
        setIsPlaying(!isPlaying);
    } else {
        setCurrentSong(song);
        setIsPlaying(true);
    }
  };

  const handleTogglePlay = () => {
    if (currentSong) {
      setIsPlaying(!isPlaying);
    }
  };

  const renderContent = () => {
    switch (activeTab) {
      case 'home':
        return <HomePage songs={mockSongs} onPlaySong={handlePlaySong} playingSongId={isPlaying ? currentSong?.id : null} />;
      case 'create':
        return <CreatePage />;
      case 'library':
        return <LibraryPage />;
      case 'more':
        return <MorePage />;
      default:
        return <HomePage songs={mockSongs} onPlaySong={handlePlaySong} playingSongId={isPlaying ? currentSong?.id : null} />;
    }
  };

  return (
    <div className="bg-[#1a1b24] min-h-screen text-gray-300 font-sans">
      <Header />
      <main className="pb-32">
        {renderContent()}
      </main>
      {currentSong && <MiniPlayer song={currentSong} isPlaying={isPlaying} onTogglePlay={handleTogglePlay} onOpenModal={() => setPlayerModalOpen(true)} />}
      <BottomNav activeTab={activeTab} setActiveTab={setActiveTab} />
      {isPlayerModalOpen && currentSong && <PlayerModal song={currentSong} isPlaying={isPlaying} onTogglePlay={handleTogglePlay} onClose={() => setPlayerModalOpen(false)} />}
    </div>
  );
};

export default App;
