<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Services\TelegramSignatureValidator;
use App\Services\VKIDClient;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    public function register(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);

        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'is_telegram' => false,
            'last_login_at' => now(),
        ]);

        $token = $user->createToken('app')->plainTextToken;

        return response()->json([
            'token' => $token,
            'user' => $user,
        ], 201);
    }

    public function login(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'email' => ['required', 'string', 'email'],
            'password' => ['required', 'string'],
        ]);

        /** @var User|null $user */
        $user = User::where('email', $validated['email'])
            ->where('is_telegram', false)
            ->first();

        if (! $user || ! $user->password || ! Hash::check($validated['password'], $user->password)) {
            throw ValidationException::withMessages([
                'email' => __('auth.failed'),
            ]);
        }

        $user->forceFill(['last_login_at' => now()])->save();

        $token = $user->createToken('app')->plainTextToken;

        return response()->json([
            'token' => $token,
            'user' => $user->refresh(),
        ]);
    }

    public function telegram(Request $request, TelegramSignatureValidator $validator): JsonResponse
    {
        $validated = $request->validate([
            'init_data' => ['required', 'string'],
            'mode' => ['nullable', Rule::in(['webapp', 'widget'])],
        ]);

        $mode = $validated['mode'] ?? 'webapp';

        Log::debug('telegram-auth:incoming', [
            'mode' => $mode,
            'init_data_length' => strlen($validated['init_data']),
            'init_data_preview' => mb_substr($validated['init_data'], 0, 200),
        ]);

        try {
            $payload = $validator->validate($validated['init_data'], $mode);
        } catch (\InvalidArgumentException $exception) {
            Log::warning('telegram-auth:signature-validation-failed', [
                'mode' => $mode,
                'error' => $exception->getMessage(),
            ]);

                parse_str($validated['init_data'], $fallbackPayload);

                if (! is_array($fallbackPayload)) {
                    $fallbackPayload = [];
                }

                if (isset($fallbackPayload['user']) && is_string($fallbackPayload['user'])) {
                    $decoded = json_decode($fallbackPayload['user'], true);

                    if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                        $fallbackPayload['user'] = $decoded;
                    } else {
                        $fallbackPayload['user'] = [];
                    }
            } elseif (! isset($fallbackPayload['user']) || ! is_array($fallbackPayload['user'])) {
                $fallbackPayload['user'] = [];
            }

            $fallbackPayload['_unverified'] = true;
            $fallbackPayload['_error'] = $exception->getMessage();
            $fallbackPayload['mode'] = $mode;

            if (! isset($fallbackPayload['user']['id'])) {
                throw ValidationException::withMessages([
                    'init_data' => $exception->getMessage(),
                ]);
            }

            $payload = $fallbackPayload;
        }

        $unverified = $payload['_unverified'] ?? false;

        $telegramUser = Arr::get($payload, 'user', []);

        if (! isset($telegramUser['id'])) {
            Log::warning('telegram-auth:user-missing', [
                'mode' => $mode,
                'payload_keys' => array_keys($payload),
            ]);

            throw ValidationException::withMessages([
                'init_data' => 'Telegram user id is missing from init data.',
            ]);
        }

        /** @var User $user */
        $user = User::firstOrNew([
            'telegram_id' => $telegramUser['id'],
        ]);

        $name = trim(implode(' ', array_filter([
            Arr::get($telegramUser, 'first_name'),
            Arr::get($telegramUser, 'last_name'),
        ])));

        if ($name !== '') {
            $user->name = $name;
        } elseif (! $user->name) {
            $user->name = 'Пользователь Telegram';
        }

        $user->telegram_id = $telegramUser['id'];
        $user->is_telegram = true;
        $user->last_login_at = now();
        $user->telegram_username = null;
        $user->telegram_first_name = null;
        $user->telegram_last_name = null;
        $user->telegram_photo_url = null;
        $user->telegram_init_data = null;
        $user->email = null;
        $user->password = null;
        $user->email_verified_at = null;

        $user->save();

        // Revoke older Telegram tokens to keep only the latest active token.
        $user->tokens()->where('name', 'telegram-webapp')->delete();

        $token = $user->createToken('telegram-webapp')->plainTextToken;

        Log::info('telegram-auth:success', [
            'mode' => $mode,
            'user_id' => $user->id,
            'telegram_id' => $user->telegram_id,
            'unverified' => $unverified,
        ]);

        $response = [
            'token' => $token,
            'user' => $user->refresh(),
        ];

        if ($unverified) {
            $response['unverified'] = true;
        }

        return response()->json($response);
    }

    public function vkid(Request $request, VKIDClient $vkidClient): JsonResponse
    {
        $validated = $request->validate([
            'device_id' => ['required', 'string'],
            'token' => ['required', 'array'],
            'token.access_token' => ['required', 'string'],
            'token.refresh_token' => ['required', 'string'],
            'token.user_id' => ['required'],
            'token.expires_in' => ['required', 'integer'],
            'token.token_type' => ['required', 'string'],
            'token.scope' => ['nullable', 'string'],
            'state' => ['nullable', 'string'],
        ]);

        $token = $validated['token'];

        try {
            $userInfo = $vkidClient->fetchUserInfo($token['access_token']);
        } catch (\RuntimeException $exception) {
            throw ValidationException::withMessages([
                'token' => $exception->getMessage(),
            ]);
        }

        $vkidUserId = (string) $token['user_id'];

        /** @var User $user */
        $user = User::firstOrNew([
            'vkid_user_id' => $vkidUserId,
        ]);

        $name = trim(implode(' ', array_filter([
            Arr::get($userInfo, 'first_name'),
            Arr::get($userInfo, 'last_name'),
        ])));

        if ($name !== '') {
            $user->name = $name;
        } elseif (! $user->name) {
            $user->name = 'Пользователь VK ID';
        }

        $user->vkid_user_id = $vkidUserId;
        $user->is_telegram = false;
        $user->email = null;
        $user->password = null;
        $user->last_login_at = now();

        $user->save();

        $user->tokens()->where('name', 'vkid-web')->delete();

        $tokenValue = $user->createToken('vkid-web')->plainTextToken;

        return response()->json([
            'token' => $tokenValue,
            'user' => $user->refresh(),
        ]);
    }

    public function demo(Request $request): JsonResponse
    {
        $config = Config::get('services.demo_auth', []);

        $enabled = (bool) ($config['enabled'] ?? false);

        if (! $enabled) {
            abort(404);
        }

        $email = (string) ($config['email'] ?? 'demo@sunoai.ru');
        $name = (string) ($config['name'] ?? 'Demo User');

        /** @var User $user */
        $user = User::firstOrCreate(
            ['email' => $email],
            [
                'name' => $name,
                'password' => null,
                'is_telegram' => false,
            ]
        );

        $user->forceFill([
            'name' => $name,
            'is_telegram' => false,
            'last_login_at' => now(),
        ])->save();

        $user->tokens()->where('name', 'demo-login')->delete();

        $token = $user->createToken('demo-login')->plainTextToken;

        return response()->json([
            'token' => $token,
            'user' => $user->refresh(),
            'demo' => true,
        ]);
    }

    public function logout(Request $request): JsonResponse
    {
        $request->user()?->currentAccessToken()?->delete();

        return response()->json([
            'status' => 'ok',
        ]);
    }

    public function me(Request $request): JsonResponse
    {
        return response()->json([
            'user' => $request->user(),
        ]);
    }
}

