<?php

namespace App\Http\Controllers;

use App\Models\Song;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Symfony\Component\HttpFoundation\Response;

class MusicGenerationController extends Controller
{
    /**
     * Запуск генерации музыки через Suno API.
     */
    public function generate(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'prompt' => ['required', 'string', 'max:500'],
            'genre' => ['nullable', 'string', 'max:120'],
            'mood' => ['nullable', 'string', 'max:120'],
            'duration' => ['nullable', 'integer', 'min:30', 'max:480'],
            'customMode' => ['sometimes', 'boolean'],
            'instrumental' => ['sometimes', 'boolean'],
            'model' => [
                'sometimes',
                'string',
                Rule::in(['V3_5', 'V4', 'V4_5', 'V4_5PLUS', 'V5']),
            ],
            'style' => ['nullable', 'string', 'max:1000'],
            'title' => ['nullable', 'string', 'max:80'],
        ]);

        $config = config('services.suno', []);

        $apiKey = $config['api_key'] ?? null;
        $baseUrl = rtrim($config['base_url'] ?? '', '/');
        $callbackUrl = $this->resolveCallbackUrl($config['callback_url'] ?? null);
        $tokenCost = (int) ($config['token_cost'] ?? 0);
        $user = $request->user();

        if ($tokenCost > 0 && $user && (($user->token_balance ?? 0) < $tokenCost)) {
            return response()->json([
                'message' => 'Недостаточно токенов для генерации. Пополните баланс.',
                'balance' => $user->token_balance ?? 0,
                'required' => $tokenCost,
            ], Response::HTTP_PAYMENT_REQUIRED);
        }

        if (!$apiKey || !$baseUrl) {
            return response()->json([
                'message' => 'Сервис Suno API не настроен. Укажите SUNO_API_KEY и SUNO_API_BASE_URL.',
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        if (!$callbackUrl) {
            return response()->json([
                'message' => 'Отсутствует SUNO_CALLBACK_URL. Укажите URL, доступный для Suno API.',
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $customMode = (bool) ($validated['customMode'] ?? false);
        $instrumental = (bool) ($validated['instrumental'] ?? false);
        $model = $validated['model'] ?? 'V5';

        $payload = [
            'customMode' => $customMode,
            'instrumental' => $instrumental,
            'model' => $model,
            'callBackUrl' => $callbackUrl,
            'prompt' => $validated['prompt'],
        ];

        if ($customMode) {
            $style = $validated['style'] ?? null;

            if (!$style) {
                $style = trim(
                    collect([$validated['genre'] ?? null, $validated['mood'] ?? null])
                        ->filter()
                        ->implode(', '),
                );
            }

            if ($style) {
                $payload['style'] = $style;
            }

            $title = $validated['title'] ?? null;
            if (!$title) {
                $title = mb_substr($validated['prompt'], 0, 80) ?: 'Generated Track';
            }

            $payload['title'] = $title;
        }

        Log::info('Sending request to Suno API', [
            'url' => "{$baseUrl}/api/v1/generate",
            'payload' => $payload,
            'has_api_key' => !empty($apiKey),
        ]);

        try {
            $response = Http::withToken($apiKey)
                ->acceptJson()
                ->post("{$baseUrl}/api/v1/generate", $payload);
            
            Log::info('Suno API response', [
                'status' => $response->status(),
                'body' => $response->body(),
                'json' => $response->json(),
            ]);
        } catch (ConnectionException $exception) {
            return response()->json([
                'message' => 'Не удалось связаться с Suno API.',
                'details' => $exception->getMessage(),
            ], Response::HTTP_BAD_GATEWAY);
        } catch (\Throwable $throwable) {
            return response()->json([
                'message' => 'Произошла ошибка при обращении к Suno API.',
                'details' => $throwable->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $responseData = $response->json();
        $responseCode = data_get($responseData, 'code');
        
        // Проверяем код ответа в JSON (Suno API может вернуть HTTP 200, но код ошибки в JSON)
        if ($response->failed() || ($responseCode !== null && $responseCode !== 200)) {
            $status = $response->status() ?: Response::HTTP_BAD_GATEWAY;
            
            // Если код ошибки в JSON, используем соответствующий HTTP статус
            if ($responseCode === 429) {
                $status = Response::HTTP_TOO_MANY_REQUESTS;
            } elseif ($responseCode === 400) {
                $status = Response::HTTP_BAD_REQUEST;
            } elseif ($responseCode === 401) {
                $status = Response::HTTP_UNAUTHORIZED;
            } elseif ($responseCode === 500) {
                $status = Response::HTTP_INTERNAL_SERVER_ERROR;
            }

            return response()->json([
                'message' => data_get($responseData, 'msg') ?? 'Suno API вернуло ошибку.',
                'code' => $responseCode,
                'data' => data_get($responseData, 'data'),
            ], $status);
        }

        $updatedUser = null;

        if ($tokenCost > 0 && $user) {
            $user->refresh();

            if (($user->token_balance ?? 0) < $tokenCost) {
                return response()->json([
                    'message' => 'Недостаточно токенов для генерации. Пополните баланс.',
                    'balance' => $user->token_balance ?? 0,
                    'required' => $tokenCost,
                ], Response::HTTP_PAYMENT_REQUIRED);
            }

            $user->forceFill([
                'token_balance' => max(0, ($user->token_balance ?? 0) - $tokenCost),
                'generation_count' => ($user->generation_count ?? 0) + 1,
            ])->save();

            $updatedUser = $user->fresh();
        }

        $generatedBy = null;
        if ($user) {
            $generatedBy = $user->name ?: trim(implode(' ', array_filter([
                $user->telegram_first_name,
                $user->telegram_last_name,
            ])));

            if ($generatedBy === '') {
                $generatedBy = null;
            }
        }

        $taskId = data_get($responseData, 'data.taskId');
        
        if (!$taskId) {
            Log::error('Suno API response missing taskId', [
                'response' => $responseData,
                'status' => $response->status(),
            ]);
            
            return response()->json([
                'message' => 'Suno API не вернуло taskId. Проверьте логи сервера.',
                'code' => $responseData['code'] ?? null,
                'msg' => $responseData['msg'] ?? null,
                'data' => $responseData['data'] ?? null,
            ], Response::HTTP_BAD_GATEWAY);
        }

        $payload = [
            'code' => $responseData['code'] ?? 200,
            'msg' => $responseData['msg'] ?? 'success',
            'data' => $responseData['data'] ?? [],
            'meta' => [
                'user_id' => $user?->id,
                'prompt' => $validated['prompt'],
                'genre' => $validated['genre'] ?? null,
                'mood' => $validated['mood'] ?? null,
                'duration' => $validated['duration'] ?? null,
                'model' => $model,
                'customMode' => $customMode,
                'instrumental' => $instrumental,
                'generated_by' => $generatedBy,
            ],
            'user' => $updatedUser,
        ];

        Cache::put($this->cacheKey($taskId), $payload, now()->addHours(12));

        return response()->json($payload);
    }

    /**
     * Продление существующего трека.
     */
    public function extend(Request $request): JsonResponse
    {
        $defaultParamFlag = $request->boolean('defaultParamFlag');

        $validated = $request->validate([
            'defaultParamFlag' => ['required', 'boolean'],
            'audioId' => ['required', 'string', 'max:255'],
            'prompt' => [
                'nullable',
                'string',
                'max:5000',
                Rule::requiredIf(fn () => $defaultParamFlag),
            ],
            'style' => [
                'nullable',
                'string',
                'max:1000',
                Rule::requiredIf(fn () => $defaultParamFlag),
            ],
            'title' => [
                'nullable',
                'string',
                'max:120',
                Rule::requiredIf(fn () => $defaultParamFlag),
            ],
            'continueAt' => [
                'nullable',
                'numeric',
                'gt:0',
                Rule::requiredIf(fn () => $defaultParamFlag),
            ],
            'personaId' => ['nullable', 'string', 'max:255'],
            'model' => [
                'required',
                'string',
                Rule::in(['V3_5', 'V4', 'V4_5', 'V4_5PLUS', 'V5']),
            ],
            'negativeTags' => ['nullable', 'string', 'max:500'],
            'vocalGender' => ['nullable', 'string', Rule::in(['m', 'f'])],
            'styleWeight' => ['nullable', 'numeric', 'min:0', 'max:1'],
            'weirdnessConstraint' => ['nullable', 'numeric', 'min:0', 'max:1'],
            'audioWeight' => ['nullable', 'numeric', 'min:0', 'max:1'],
        ]);

        $config = config('services.suno', []);

        $apiKey = $config['api_key'] ?? null;
        $baseUrl = rtrim($config['base_url'] ?? '', '/');
        $callbackUrl = $this->resolveCallbackUrl($config['callback_url'] ?? null);

        if (!$apiKey || !$baseUrl) {
            return response()->json([
                'message' => 'Сервис Suno API не настроен. Укажите SUNO_API_KEY и SUNO_API_BASE_URL.',
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        if (!$callbackUrl) {
            return response()->json([
                'message' => 'Отсутствует SUNO_CALLBACK_URL. Укажите URL, доступный для Suno API.',
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $payload = [
            'defaultParamFlag' => $defaultParamFlag,
            'audioId' => $validated['audioId'],
            'model' => $validated['model'],
            'callBackUrl' => $callbackUrl,
        ];

        if ($defaultParamFlag) {
            $payload['prompt'] = trim((string) $validated['prompt']);
            $payload['style'] = trim((string) $validated['style']);
            $payload['title'] = trim((string) $validated['title']);
            if ($validated['continueAt'] !== null) {
                $payload['continueAt'] = (float) $validated['continueAt'];
            }
        }

        foreach (['personaId', 'negativeTags', 'vocalGender'] as $key) {
            if (!empty($validated[$key])) {
                $payload[$key] = $validated[$key];
            }
        }

        foreach (['styleWeight', 'weirdnessConstraint', 'audioWeight'] as $key) {
            if ($validated[$key] !== null) {
                $payload[$key] = (float) $validated[$key];
            }
        }

        try {
            $response = Http::withToken($apiKey)
                ->acceptJson()
                ->post("{$baseUrl}/api/v1/generate/extend", $payload);
        } catch (ConnectionException $exception) {
            return response()->json([
                'message' => 'Не удалось связаться с Suno API.',
                'details' => $exception->getMessage(),
            ], Response::HTTP_BAD_GATEWAY);
        } catch (\Throwable $throwable) {
            return response()->json([
                'message' => 'Произошла ошибка при обращении к Suno API.',
                'details' => $throwable->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        if ($response->failed()) {
            $status = $response->status() ?: Response::HTTP_BAD_GATEWAY;

            return response()->json([
                'message' => $response->json('msg') ?? 'Suno API вернуло ошибку.',
                'code' => $response->json('code'),
                'data' => $response->json('data'),
            ], $status);
        }

        $user = $request->user();
        $generatedBy = null;
        if ($user) {
            $generatedBy = $user->name ?: trim(implode(' ', array_filter([
                $user->telegram_first_name,
                $user->telegram_last_name,
            ])));

            if ($generatedBy === '') {
                $generatedBy = null;
            }
        }

        $payloadResponse = [
            'code' => $response->json('code'),
            'msg' => $response->json('msg'),
            'data' => $response->json('data'),
            'meta' => [
                'type' => 'extend',
                'audioId' => $validated['audioId'],
                'model' => $validated['model'],
                'defaultParamFlag' => $defaultParamFlag,
                'prompt' => $validated['prompt'] ?? null,
                'style' => $validated['style'] ?? null,
                'title' => $validated['title'] ?? null,
                'continueAt' => $validated['continueAt'] ?? null,
                'generated_by' => $generatedBy,
            ],
        ];

        $taskId = data_get($payloadResponse, 'data.taskId');
        if ($taskId) {
            Cache::put($this->cacheKey($taskId), $payloadResponse, now()->addHours(12));
        }

        return response()->json($payloadResponse);
    }

    /**
     * Получение статуса генерации по taskId.
     */
    public function show(Request $request, string $taskId): JsonResponse
    {
        $config = config('services.suno', []);

        $apiKey = $config['api_key'] ?? null;
        $baseUrl = rtrim($config['base_url'] ?? '', '/');

        $cacheKey = $this->cacheKey($taskId);
        $cached = Cache::get($cacheKey);

        $skipCache = $request->boolean('refresh') || $request->boolean('skip_cache');

        if ($cached && !$skipCache) {
            return response()->json($cached);
        }

        if (!$apiKey || !$baseUrl) {
            return response()->json([
                'message' => 'Сервис Suno API не настроен. Укажите SUNO_API_KEY и SUNO_API_BASE_URL.',
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        try {
            $response = Http::withToken($apiKey)
                ->acceptJson()
                ->get("{$baseUrl}/api/v1/generate/" . urlencode($taskId));
        } catch (ConnectionException $exception) {
            if ($cached) {
                return response()->json($cached);
            }

            return response()->json([
                'message' => 'Не удалось получить статус задачи Suno.',
                'details' => $exception->getMessage(),
            ], Response::HTTP_BAD_GATEWAY);
        } catch (\Throwable $throwable) {
            if ($cached) {
                return response()->json($cached);
            }

            return response()->json([
                'message' => 'Произошла ошибка при обновлении статуса задачи Suno.',
                'details' => $throwable->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        if ($response->failed()) {
            $status = $response->status() ?: Response::HTTP_BAD_GATEWAY;

            if ($cached) {
                return response()->json($cached);
            }

            return response()->json([
                'message' => $response->json('msg') ?? 'Suno API вернуло ошибку.',
                'code' => $response->json('code'),
                'data' => $response->json('data'),
            ], $status);
        }

        $payload = [
            'code' => $response->json('code'),
            'msg' => $response->json('msg'),
            'data' => $response->json('data'),
        ];

        $existing = Cache::get($cacheKey, []);
        if (isset($existing['meta']) && !isset($payload['meta'])) {
            $payload['meta'] = $existing['meta'];
        }

        Cache::put($cacheKey, $payload, now()->addHours(12));

        return response()->json($payload);
    }

    /**
     * Обработка callback-уведомлений Suno (необязательно).
     */
    public function callback(Request $request): JsonResponse
    {
        $payload = $request->all();

        $code = data_get($payload, 'code');
        $taskId = data_get($payload, 'data.task_id');
        $callbackType = data_get($payload, 'data.callbackType');

        if (!$code || !$taskId) {
            Log::warning('Получен неверный callback от Suno API', [
                'payload' => $payload,
            ]);

            return response()->json([
                'message' => 'Некорректный формат callback.',
            ], Response::HTTP_BAD_REQUEST);
        }

        $cacheKey = $this->cacheKey($taskId);

        $normalized = [
            'code' => $code,
            'msg' => data_get($payload, 'msg'),
            'data' => data_get($payload, 'data'),
        ];

        $existing = Cache::get($cacheKey, []);

        if (!empty($existing['meta'])) {
            $normalized['meta'] = $existing['meta'];
        }

        $normalized['meta'] = array_merge(
            $normalized['meta'] ?? [],
            [
                'callbackType' => $callbackType,
                'callbackReceivedAt' => now()->toIso8601String(),
            ],
        );

        if ($code !== 200) {
            $normalized['meta']['callbackError'] = $payload['msg'] ?? null;
        }

        Cache::put($cacheKey, $normalized, now()->addHours(12));

        Log::info('Callback от Suno API получен', [
            'taskId' => $taskId,
            'callbackType' => $callbackType,
            'code' => $code,
            'hasData' => !empty(data_get($normalized, 'data.data')),
        ]);

        // Сохраняем треки в БД при завершении генерации
        if ($callbackType === 'complete' && $code === 200) {
            $this->saveSongsToDatabase($taskId, $normalized, $existing);
        }

        return response()->json(['status' => 'received']);
    }

    protected function cacheKey(string $taskId): string
    {
        return 'suno:generation:' . $taskId;
    }

    protected function resolveCallbackUrl(?string $configuredUrl): ?string
    {
        if ($configuredUrl) {
            return $configuredUrl;
        }

        try {
            return route('music.callback', [], true);
        } catch (\Throwable $throwable) {
            Log::warning('Не удалось сформировать URL для music.callback', [
                'error' => $throwable->getMessage(),
            ]);
        }

        return null;
    }

    /**
     * Сохранить треки в базу данных.
     */
    protected function saveSongsToDatabase(string $taskId, array $normalized, array $existing): void
    {
        try {
            Log::info('Saving songs to database', [
                'taskId' => $taskId,
                'normalized' => $normalized,
            ]);

            $data = data_get($normalized, 'data', []);
            $songsData = data_get($data, 'data', []);
            
            Log::info('Extracted songs data', [
                'taskId' => $taskId,
                'songsCount' => is_array($songsData) ? count($songsData) : 0,
                'songsData' => $songsData,
            ]);
            
            if (!is_array($songsData) || empty($songsData)) {
                Log::warning('Нет данных о треках для сохранения', [
                    'taskId' => $taskId,
                    'data' => $data,
                    'normalized' => $normalized,
                ]);
                return;
            }

            // Получаем user_id из метаданных
            $userId = data_get($existing, 'meta.user_id') ?? data_get($normalized, 'meta.user_id');
            
            if (!$userId) {
                Log::warning('Не найден user_id для сохранения треков', [
                    'taskId' => $taskId,
                    'existing' => $existing,
                    'normalized' => $normalized,
                ]);
                return;
            }

            $meta = data_get($normalized, 'meta', []);
            $model = $meta['model'] ?? 'V5';

            foreach ($songsData as $songData) {
                $sunoSongId = data_get($songData, 'id');
                
                if (!$sunoSongId) {
                    Log::warning('Трек без ID пропущен', [
                        'taskId' => $taskId,
                        'songData' => $songData,
                    ]);
                    continue;
                }

                // Проверяем, существует ли уже трек
                $existingSong = Song::where('suno_song_id', $sunoSongId)
                    ->orWhere(function ($query) use ($taskId, $sunoSongId) {
                        $query->where('task_id', $taskId)
                              ->where('suno_song_id', $sunoSongId);
                    })
                    ->first();

                // Используем prompt из самого трека (как в документации)
                $prompt = data_get($songData, 'prompt') ?? $meta['prompt'] ?? null;
                
                // Используем правильные названия полей из документации
                $songAttributes = [
                    'user_id' => $userId,
                    'suno_song_id' => $sunoSongId,
                    'task_id' => $taskId,
                    'title' => data_get($songData, 'title') ?? 'Untitled',
                    'artist' => data_get($songData, 'artist'),
                    'prompt' => $prompt,
                    'tags' => data_get($songData, 'tags'),
                    'artwork_url' => data_get($songData, 'image_url'),
                    'source_artwork_url' => data_get($songData, 'source_image_url') ?? data_get($songData, 'image_url'),
                    'audio_url' => data_get($songData, 'audio_url'),
                    'source_audio_url' => data_get($songData, 'source_audio_url') ?? data_get($songData, 'audio_url'),
                    'stream_url' => data_get($songData, 'stream_audio_url') ?? data_get($songData, 'audio_url'),
                    'source_stream_url' => data_get($songData, 'source_stream_audio_url') ?? data_get($songData, 'stream_audio_url') ?? data_get($songData, 'audio_url'),
                    'source_audio_id' => $sunoSongId,
                    'model' => data_get($songData, 'model_name') ?? $model,
                    'duration_seconds' => (int) (data_get($songData, 'duration') ?? 0),
                    'status' => 'complete',
                ];

                if ($existingSong) {
                    $existingSong->update($songAttributes);
                    Log::info('Трек обновлен в БД', [
                        'song_id' => $existingSong->id,
                        'suno_song_id' => $sunoSongId,
                        'title' => $songAttributes['title'],
                        'audio_url' => $songAttributes['audio_url'],
                    ]);
                } else {
                    $song = Song::create($songAttributes);
                    Log::info('Трек сохранен в БД', [
                        'song_id' => $song->id,
                        'suno_song_id' => $sunoSongId,
                        'title' => $songAttributes['title'],
                        'audio_url' => $songAttributes['audio_url'],
                    ]);
                }
            }
        } catch (\Throwable $throwable) {
            Log::error('Ошибка при сохранении треков в БД', [
                'taskId' => $taskId,
                'error' => $throwable->getMessage(),
                'trace' => $throwable->getTraceAsString(),
            ]);
        }
    }

    /**
     * Генерация трека по образцу (cover).
     */
    public function cover(Request $request): JsonResponse
    {
        $user = $request->user();
        $apiKey = config('services.suno.api_key');
        $baseUrl = config('services.suno.base_url', 'https://api.sunoapi.org');
        $config = config('services.suno', []);
        $callbackUrl = $this->resolveCallbackUrl($config['callback_url'] ?? null);

        if (!$callbackUrl) {
            return response()->json([
                'message' => 'Не настроен URL для callback.',
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $validated = $request->validate([
            'audio_file' => ['required', 'file', 'mimes:mp3,wav,m4a', 'max:10240'], // 10MB max
            'prompt' => ['nullable', 'string', 'max:5000'],
            'title' => ['nullable', 'string', 'max:255'],
            'tags' => ['nullable', 'string', 'max:1000'],
            'model' => ['nullable', 'string', Rule::in(['V3_5', 'V4', 'V4_5', 'V4_5PLUS', 'V5'])],
        ]);

        $generatedBy = null;
        if ($user) {
            $generatedBy = $user->name ?: trim(implode(' ', array_filter([
                $user->telegram_first_name,
                $user->telegram_last_name,
            ])));
            if ($generatedBy === '') {
                $generatedBy = null;
            }
        }

        // Загружаем файл
        $audioFile = $request->file('audio_file');
        $audioPath = $audioFile->getRealPath();
        $audioName = $audioFile->getClientOriginalName();

        $payload = [
            'callBackUrl' => $callbackUrl,
            'prompt' => $validated['prompt'] ?? null,
            'title' => $validated['title'] ?? null,
            'tags' => $validated['tags'] ?? null,
            'model' => $validated['model'] ?? 'V5',
        ];

        Log::info('Sending cover request to Suno API', [
            'url' => "{$baseUrl}/api/v1/generate/cover",
            'payload' => $payload,
            'has_api_key' => !empty($apiKey),
        ]);

        try {
            $response = Http::withToken($apiKey)
                ->attach('audio_file', file_get_contents($audioPath), $audioName)
                ->acceptJson()
                ->post("{$baseUrl}/api/v1/generate/cover", $payload);
            
            Log::info('Suno API cover response', [
                'status' => $response->status(),
                'body' => $response->body(),
                'json' => $response->json(),
            ]);
        } catch (ConnectionException $exception) {
            return response()->json([
                'message' => 'Не удалось связаться с Suno API.',
                'details' => $exception->getMessage(),
            ], Response::HTTP_BAD_GATEWAY);
        } catch (\Throwable $throwable) {
            return response()->json([
                'message' => 'Произошла ошибка при обращении к Suno API.',
                'details' => $throwable->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $responseData = $response->json();
        $responseCode = data_get($responseData, 'code');
        
        if ($response->failed() || ($responseCode !== null && $responseCode !== 200)) {
            $status = $response->status() ?: Response::HTTP_BAD_GATEWAY;
            
            if ($responseCode === 429) {
                $status = Response::HTTP_TOO_MANY_REQUESTS;
            } elseif ($responseCode === 400) {
                $status = Response::HTTP_BAD_REQUEST;
            } elseif ($responseCode === 401) {
                $status = Response::HTTP_UNAUTHORIZED;
            } elseif ($responseCode === 500) {
                $status = Response::HTTP_INTERNAL_SERVER_ERROR;
            }

            return response()->json([
                'message' => data_get($responseData, 'msg') ?? 'Suno API вернуло ошибку.',
                'code' => $responseCode,
                'data' => data_get($responseData, 'data'),
            ], $status);
        }

        $taskId = data_get($responseData, 'data.taskId');
        
        if (!$taskId) {
            Log::error('Suno API cover response missing taskId', [
                'response' => $responseData,
                'status' => $response->status(),
            ]);
            
            return response()->json([
                'message' => 'Suno API не вернуло taskId. Проверьте логи сервера.',
                'code' => $responseData['code'] ?? null,
                'msg' => $responseData['msg'] ?? null,
                'data' => $responseData['data'] ?? null,
            ], Response::HTTP_BAD_GATEWAY);
        }

        $payload = [
            'code' => $responseData['code'] ?? 200,
            'msg' => $responseData['msg'] ?? 'success',
            'data' => $responseData['data'] ?? [],
            'meta' => [
                'user_id' => $user?->id,
                'prompt' => $validated['prompt'] ?? null,
                'title' => $validated['title'] ?? null,
                'tags' => $validated['tags'] ?? null,
                'model' => $validated['model'] ?? 'V5',
                'type' => 'cover',
                'generated_by' => $generatedBy,
            ],
        ];

        Cache::put($this->cacheKey($taskId), $payload, now()->addHours(12));

        return response()->json($payload);
    }

    /**
     * Добавление инструментала к существующей песне.
     */
    public function addInstrumental(Request $request): JsonResponse
    {
        $user = $request->user();
        $apiKey = config('services.suno.api_key');
        $baseUrl = config('services.suno.base_url', 'https://api.sunoapi.org');
        $config = config('services.suno', []);
        $callbackUrl = $this->resolveCallbackUrl($config['callback_url'] ?? null);

        if (!$callbackUrl) {
            return response()->json([
                'message' => 'Не настроен URL для callback.',
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $validated = $request->validate([
            'audioId' => ['required', 'string', 'max:255'],
            'prompt' => ['nullable', 'string', 'max:5000'],
            'style' => ['nullable', 'string', 'max:1000'],
        ]);

        $generatedBy = null;
        if ($user) {
            $generatedBy = $user->name ?: trim(implode(' ', array_filter([
                $user->telegram_first_name,
                $user->telegram_last_name,
            ])));
            if ($generatedBy === '') {
                $generatedBy = null;
            }
        }

        $payload = [
            'callBackUrl' => $callbackUrl,
            'audioId' => $validated['audioId'],
            'prompt' => $validated['prompt'] ?? null,
            'style' => $validated['style'] ?? null,
        ];

        Log::info('Sending add-instrumental request to Suno API', [
            'url' => "{$baseUrl}/api/v1/generate/add-instrumental",
            'payload' => $payload,
            'has_api_key' => !empty($apiKey),
        ]);

        try {
            $response = Http::withToken($apiKey)
                ->acceptJson()
                ->post("{$baseUrl}/api/v1/generate/add-instrumental", $payload);
            
            Log::info('Suno API add-instrumental response', [
                'status' => $response->status(),
                'body' => $response->body(),
                'json' => $response->json(),
            ]);
        } catch (ConnectionException $exception) {
            return response()->json([
                'message' => 'Не удалось связаться с Suno API.',
                'details' => $exception->getMessage(),
            ], Response::HTTP_BAD_GATEWAY);
        } catch (\Throwable $throwable) {
            return response()->json([
                'message' => 'Произошла ошибка при обращении к Suno API.',
                'details' => $throwable->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $responseData = $response->json();
        $responseCode = data_get($responseData, 'code');
        
        if ($response->failed() || ($responseCode !== null && $responseCode !== 200)) {
            $status = $response->status() ?: Response::HTTP_BAD_GATEWAY;
            
            if ($responseCode === 429) {
                $status = Response::HTTP_TOO_MANY_REQUESTS;
            } elseif ($responseCode === 400) {
                $status = Response::HTTP_BAD_REQUEST;
            } elseif ($responseCode === 401) {
                $status = Response::HTTP_UNAUTHORIZED;
            } elseif ($responseCode === 500) {
                $status = Response::HTTP_INTERNAL_SERVER_ERROR;
            }

            return response()->json([
                'message' => data_get($responseData, 'msg') ?? 'Suno API вернуло ошибку.',
                'code' => $responseCode,
                'data' => data_get($responseData, 'data'),
            ], $status);
        }

        $taskId = data_get($responseData, 'data.taskId');
        
        if (!$taskId) {
            Log::error('Suno API add-instrumental response missing taskId', [
                'response' => $responseData,
                'status' => $response->status(),
            ]);
            
            return response()->json([
                'message' => 'Suno API не вернуло taskId. Проверьте логи сервера.',
                'code' => $responseData['code'] ?? null,
                'msg' => $responseData['msg'] ?? null,
                'data' => $responseData['data'] ?? null,
            ], Response::HTTP_BAD_GATEWAY);
        }

        $payload = [
            'code' => $responseData['code'] ?? 200,
            'msg' => $responseData['msg'] ?? 'success',
            'data' => $responseData['data'] ?? [],
            'meta' => [
                'user_id' => $user?->id,
                'audioId' => $validated['audioId'],
                'prompt' => $validated['prompt'] ?? null,
                'style' => $validated['style'] ?? null,
                'type' => 'add-instrumental',
                'generated_by' => $generatedBy,
            ],
        ];

        Cache::put($this->cacheKey($taskId), $payload, now()->addHours(12));

        return response()->json($payload);
    }

    /**
     * Добавление голоса к существующей песне.
     */
    public function addVocals(Request $request): JsonResponse
    {
        $user = $request->user();
        $apiKey = config('services.suno.api_key');
        $baseUrl = config('services.suno.base_url', 'https://api.sunoapi.org');
        $config = config('services.suno', []);
        $callbackUrl = $this->resolveCallbackUrl($config['callback_url'] ?? null);

        if (!$callbackUrl) {
            return response()->json([
                'message' => 'Не настроен URL для callback.',
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $validated = $request->validate([
            'audioId' => ['required', 'string', 'max:255'],
            'prompt' => ['nullable', 'string', 'max:5000'],
            'style' => ['nullable', 'string', 'max:1000'],
            'lyrics' => ['nullable', 'string', 'max:5000'],
            'title' => ['nullable', 'string', 'max:255'],
            'tags' => ['nullable', 'string', 'max:1000'],
        ]);

        $generatedBy = null;
        if ($user) {
            $generatedBy = $user->name ?: trim(implode(' ', array_filter([
                $user->telegram_first_name,
                $user->telegram_last_name,
            ])));
            if ($generatedBy === '') {
                $generatedBy = null;
            }
        }

        $payload = [
            'callBackUrl' => $callbackUrl,
            'audioId' => $validated['audioId'],
            'prompt' => $validated['prompt'] ?? null,
            'style' => $validated['style'] ?? null,
            'lyrics' => $validated['lyrics'] ?? null,
            'title' => $validated['title'] ?? null,
            'tags' => $validated['tags'] ?? null,
        ];

        Log::info('Sending add-vocals request to Suno API', [
            'url' => "{$baseUrl}/api/v1/generate/add-vocals",
            'payload' => $payload,
            'has_api_key' => !empty($apiKey),
        ]);

        try {
            $response = Http::withToken($apiKey)
                ->acceptJson()
                ->post("{$baseUrl}/api/v1/generate/add-vocals", $payload);
            
            Log::info('Suno API add-vocals response', [
                'status' => $response->status(),
                'body' => $response->body(),
                'json' => $response->json(),
            ]);
        } catch (ConnectionException $exception) {
            return response()->json([
                'message' => 'Не удалось связаться с Suno API.',
                'details' => $exception->getMessage(),
            ], Response::HTTP_BAD_GATEWAY);
        } catch (\Throwable $throwable) {
            return response()->json([
                'message' => 'Произошла ошибка при обращении к Suno API.',
                'details' => $throwable->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        $responseData = $response->json();
        $responseCode = data_get($responseData, 'code');
        
        if ($response->failed() || ($responseCode !== null && $responseCode !== 200)) {
            $status = $response->status() ?: Response::HTTP_BAD_GATEWAY;
            
            if ($responseCode === 429) {
                $status = Response::HTTP_TOO_MANY_REQUESTS;
            } elseif ($responseCode === 400) {
                $status = Response::HTTP_BAD_REQUEST;
            } elseif ($responseCode === 401) {
                $status = Response::HTTP_UNAUTHORIZED;
            } elseif ($responseCode === 500) {
                $status = Response::HTTP_INTERNAL_SERVER_ERROR;
            }

            return response()->json([
                'message' => data_get($responseData, 'msg') ?? 'Suno API вернуло ошибку.',
                'code' => $responseCode,
                'data' => data_get($responseData, 'data'),
            ], $status);
        }

        $taskId = data_get($responseData, 'data.taskId');
        
        if (!$taskId) {
            Log::error('Suno API add-vocals response missing taskId', [
                'response' => $responseData,
                'status' => $response->status(),
            ]);
            
            return response()->json([
                'message' => 'Suno API не вернуло taskId. Проверьте логи сервера.',
                'code' => $responseData['code'] ?? null,
                'msg' => $responseData['msg'] ?? null,
                'data' => $responseData['data'] ?? null,
            ], Response::HTTP_BAD_GATEWAY);
        }

        $payload = [
            'code' => $responseData['code'] ?? 200,
            'msg' => $responseData['msg'] ?? 'success',
            'data' => $responseData['data'] ?? [],
            'meta' => [
                'user_id' => $user?->id,
                'audioId' => $validated['audioId'],
                'prompt' => $validated['prompt'] ?? null,
                'style' => $validated['style'] ?? null,
                'lyrics' => $validated['lyrics'] ?? null,
                'title' => $validated['title'] ?? null,
                'tags' => $validated['tags'] ?? null,
                'type' => 'add-vocals',
                'generated_by' => $generatedBy,
            ],
        ];

        Cache::put($this->cacheKey($taskId), $payload, now()->addHours(12));

        return response()->json($payload);
    }
}

