<?php

namespace App\Http\Controllers;

use App\Models\Song;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class PublicSongsController extends Controller
{
    /**
     * Get public songs with optional filtering.
     */
    public function index(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'filter' => ['nullable', 'string', 'in:editorial,today,week,month,all,live'],
            'limit' => ['nullable', 'integer', 'min:1', 'max:100'],
            'offset' => ['nullable', 'integer', 'min:0'],
        ]);

        $filter = $validated['filter'] ?? 'all';
        $limit = $validated['limit'] ?? 50;
        $offset = $validated['offset'] ?? 0;

        $query = Song::public()->whereNotNull('published_at');

        // Map filter names from frontend to backend
        $periodMap = [
            'editorial' => 'all', // Выбор редакции - показываем все
            'today' => 'today',
            'week' => 'week',
            'month' => 'month',
            'all' => 'all',
            'live' => 'all', // В эфире - показываем все публичные
        ];

        $period = $periodMap[$filter] ?? 'all';
        
        if ($period !== 'all') {
            $query->inPeriod($period);
        }

        $songs = $query
            ->orderBy('published_at', 'desc')
            ->offset($offset)
            ->limit($limit)
            ->with('user:id,name,telegram_username')
            ->get()
            ->map(function ($song) {
                return [
                    'id' => (string) $song->suno_song_id,
                    'taskId' => $song->task_id,
                    'title' => $song->title,
                    'artist' => $song->artist ?? ($song->user->name ?? 'Suno AI'),
                    'prompt' => $song->prompt,
                    'tags' => $song->tags ? json_decode($song->tags, true) : null,
                    'artworkUrl' => $song->artwork_url,
                    'sourceArtworkUrl' => $song->source_artwork_url,
                    'audioUrl' => $song->audio_url,
                    'sourceAudioUrl' => $song->source_audio_url,
                    'streamUrl' => $song->stream_url,
                    'sourceStreamUrl' => $song->source_stream_url,
                    'sourceAudioId' => $song->source_audio_id,
                    'status' => $song->status,
                    'model' => $song->model,
                    'durationSeconds' => $song->duration_seconds,
                    'createdAt' => $song->created_at->toIso8601String(),
                    'publishedAt' => $song->published_at?->toIso8601String(),
                    'playsCount' => $song->plays_count,
                    'likesCount' => $song->likes_count,
                    'isPublic' => $song->is_public,
                ];
            });

        return response()->json([
            'songs' => $songs,
            'total' => $query->count(),
            'filter' => $filter,
        ]);
    }

    /**
     * Publish a song (make it public).
     */
    public function publish(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'song_id' => ['required', 'string'],
            'task_id' => ['nullable', 'string'],
        ]);

        $songId = $validated['song_id'];
        $taskId = $validated['task_id'] ?? null;
        $user = $request->user();

        if (!$user) {
            return response()->json([
                'message' => 'Требуется авторизация.',
            ], 401);
        }

        // Find or create song
        $song = Song::firstOrNew([
            'suno_song_id' => $songId,
            'user_id' => $user->id,
        ]);

        // If song doesn't exist, we need basic info
        if (!$song->exists) {
            // Try to get from cache
            if ($taskId) {
                $cacheKey = 'suno:generation:' . $taskId;
                $cached = \Illuminate\Support\Facades\Cache::get($cacheKey);
                
                if ($cached && isset($cached['data']['data'])) {
                    $songsData = is_array($cached['data']['data']) 
                        ? $cached['data']['data'] 
                        : [];
                    
                    $songData = collect($songsData)->firstWhere('songId', $songId) 
                        ?? collect($songsData)->first();
                    
                    if ($songData) {
                        $song->title = $songData['title'] ?? 'Без названия';
                        $song->artist = $songData['artist'] ?? $user->name ?? 'Suno AI';
                        $song->artwork_url = $songData['imageUrl'] ?? $songData['image_url'] ?? null;
                        $song->audio_url = $songData['audioUrl'] ?? $songData['audio_url'] ?? null;
                        $song->stream_url = $songData['streamUrl'] ?? $songData['stream_audio_url'] ?? null;
                        $song->status = $songData['status'] ?? 'complete';
                        $song->model = $songData['model'] ?? null;
                        $song->duration_seconds = $songData['duration'] ?? $songData['durationSeconds'] ?? null;
                    }
                }
            }
            
            if (!$song->title) {
                $song->title = 'Без названия';
            }
            $song->task_id = $taskId;
        }

        // Mark as public
        $song->is_public = true;
        $song->published_at = now();
        $song->save();

        return response()->json([
            'message' => 'Трек опубликован.',
            'song' => [
                'id' => (string) $song->suno_song_id,
                'isPublic' => $song->is_public,
                'publishedAt' => $song->published_at->toIso8601String(),
            ],
        ]);
    }

    /**
     * Unpublish a song (make it private).
     */
    public function unpublish(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'song_id' => ['required', 'string'],
        ]);

        $songId = $validated['song_id'];
        $user = $request->user();

        if (!$user) {
            return response()->json([
                'message' => 'Требуется авторизация.',
            ], 401);
        }

        $song = Song::where('suno_song_id', $songId)
            ->where('user_id', $user->id)
            ->first();

        if (!$song) {
            return response()->json([
                'message' => 'Трек не найден.',
            ], 404);
        }

        $song->is_public = false;
        $song->save();

        return response()->json([
            'message' => 'Трек снят с публикации.',
        ]);
    }
}
