import React, { useState, useEffect, useRef } from 'react';
import { SpinnerIcon } from './Icons';
import { GenerationTask, Song } from '../types';

const genres = ['Электроника', 'Хип-хоп', 'Поп', 'Рок', 'Джаз', 'Классика'];

const Chip: React.FC<{ label: string; isSelected: boolean; onClick: () => void; disabled: boolean }> = ({ label, isSelected, onClick, disabled }) => (
  <button
    onClick={onClick}
    disabled={disabled}
    className={`px-4 py-2 text-sm font-semibold rounded-full transition-colors whitespace-nowrap ${
      isSelected ? 'bg-purple-600 text-white' : 'bg-[#30313d] text-gray-300'
    } ${disabled ? 'opacity-50 cursor-not-allowed' : 'hover:bg-purple-500'}`}
  >
    {label}
  </button>
);

interface CreatePageProps {
  isAuthenticated: boolean;
  onRequireAuth: () => void;
  onGenerate: (payload: {
    prompt: string;
    genre?: string | null;
    mood?: string | null;
    duration?: number;
    customMode?: boolean;
    instrumental?: boolean;
    model?: 'V3_5' | 'V4' | 'V4_5' | 'V4_5PLUS' | 'V5';
  }) => Promise<void>;
  onCover?: (audioFile: File, options: { prompt?: string; title?: string; tags?: string; model?: string }) => Promise<void>;
  onAddInstrumental?: (audioId: string, options: { prompt?: string; style?: string }) => Promise<void>;
  onAddVocals?: (audioId: string, options: { prompt?: string; style?: string; lyrics?: string; title?: string; tags?: string }) => Promise<void>;
  isGenerating: boolean;
  latestTask: GenerationTask | null;
  generationError: string | null;
  onExtendSong: (song: Song) => Promise<void>;
  extendingSongId: string | null;
  songs?: Song[];
}

const CreatePage: React.FC<CreatePageProps> = ({
  isAuthenticated,
  onRequireAuth,
  onGenerate,
  onCover,
  onAddInstrumental,
  onAddVocals,
  isGenerating,
  latestTask,
  generationError,
  onExtendSong,
  extendingSongId,
  songs = [],
}) => {
  const [activeTab, setActiveTab] = useState<'generate' | 'cover' | 'instrumental' | 'vocals'>('generate');
  const [selectedGenre, setSelectedGenre] = useState<string | null>(genres[0]);
  const [prompt, setPrompt] = useState('');
  const [formError, setFormError] = useState<string | null>(null);
  const audioRefs = useRef<Map<string, HTMLAudioElement>>(new Map());
  
  // Cover state
  const [coverFile, setCoverFile] = useState<File | null>(null);
  const [coverPrompt, setCoverPrompt] = useState('');
  const [coverTitle, setCoverTitle] = useState('');
  const [coverTags, setCoverTags] = useState('');
  const [coverModel, setCoverModel] = useState<'V5'>('V5');
  
  // Instrumental state
  const [selectedSongForInstrumental, setSelectedSongForInstrumental] = useState<Song | null>(null);
  const [instrumentalPrompt, setInstrumentalPrompt] = useState('');
  const [instrumentalStyle, setInstrumentalStyle] = useState('');
  
  // Vocals state
  const [selectedSongForVocals, setSelectedSongForVocals] = useState<Song | null>(null);
  const [vocalsPrompt, setVocalsPrompt] = useState('');
  const [vocalsStyle, setVocalsStyle] = useState('');
  const [vocalsLyrics, setVocalsLyrics] = useState('');
  const [vocalsTitle, setVocalsTitle] = useState('');
  const [vocalsTags, setVocalsTags] = useState('');

  const handleGenerate = async () => {
    if (!isAuthenticated) {
      onRequireAuth();
      return;
    }

    setFormError(null);

    const trimmedPrompt = prompt.trim();
    const fallbackPromptParts: string[] = ['создай трек'];

    if (selectedGenre) {
      fallbackPromptParts.push(`в жанре ${selectedGenre.toLowerCase()}`);
    }

    const fallbackPrompt = fallbackPromptParts.join(' ');

    const finalPrompt = (trimmedPrompt.length > 0 ? trimmedPrompt : fallbackPrompt).slice(0, 500);

    if (finalPrompt.length === 0) {
      setFormError('Опишите желаемую музыку или выберите параметры для генерации.');
      return;
    }

    try {
      await onGenerate({
        prompt: finalPrompt,
        genre: selectedGenre,
        customMode: false,
        instrumental: false,
        model: 'V5',
      });
    } catch (error) {
      const message = error instanceof Error ? error.message : 'Не удалось запустить генерацию музыки.';
      setFormError(message);
    }
  };

  const handleExtend = async (song: Song) => {
    if (!isAuthenticated) {
      onRequireAuth();
      return;
    }

    setFormError(null);

    try {
      await onExtendSong(song);
    } catch (error) {
      const message = error instanceof Error ? error.message : 'Не удалось продлить трек.';
      setFormError(message);
    }
  };

  // Автоматически запускаем воспроизведение когда трек становится готовым
  useEffect(() => {
    if (!latestTask) return;

    latestTask.songs.forEach((song) => {
      const audioUrl = song.streamUrl ?? song.audioUrl ?? song.sourceStreamUrl ?? song.sourceAudioUrl;
      if (audioUrl) {
        const audioElement = audioRefs.current.get(song.id);
        if (audioElement && audioElement.paused && audioElement.readyState >= 2) {
          audioElement.play().catch((err) => {
            console.log('Auto-play prevented for song:', song.id, err);
          });
        }
      }
    });
  }, [latestTask]);

  const handleCover = async () => {
    if (!isAuthenticated) {
      onRequireAuth();
      return;
    }
    if (!coverFile || !onCover) {
      setFormError('Выберите аудио файл для генерации cover.');
      return;
    }
    setFormError(null);
    try {
      await onCover(coverFile, {
        prompt: coverPrompt || undefined,
        title: coverTitle || undefined,
        tags: coverTags || undefined,
        model: coverModel,
      });
    } catch (error) {
      const message = error instanceof Error ? error.message : 'Не удалось запустить генерацию cover.';
      setFormError(message);
    }
  };

  const handleAddInstrumental = async () => {
    if (!isAuthenticated) {
      onRequireAuth();
      return;
    }
    if (!selectedSongForInstrumental || !onAddInstrumental) {
      setFormError('Выберите трек для добавления инструментала.');
      return;
    }
    const audioId = selectedSongForInstrumental.id;
    if (!audioId) {
      setFormError('У выбранного трека нет ID.');
      return;
    }
    setFormError(null);
    try {
      await onAddInstrumental(audioId, {
        prompt: instrumentalPrompt || undefined,
        style: instrumentalStyle || undefined,
      });
    } catch (error) {
      const message = error instanceof Error ? error.message : 'Не удалось добавить инструментал.';
      setFormError(message);
    }
  };

  const handleAddVocals = async () => {
    if (!isAuthenticated) {
      onRequireAuth();
      return;
    }
    if (!selectedSongForVocals || !onAddVocals) {
      setFormError('Выберите трек для добавления голоса.');
      return;
    }
    const audioId = selectedSongForVocals.id;
    if (!audioId) {
      setFormError('У выбранного трека нет ID.');
      return;
    }
    setFormError(null);
    try {
      await onAddVocals(audioId, {
        prompt: vocalsPrompt || undefined,
        style: vocalsStyle || undefined,
        lyrics: vocalsLyrics || undefined,
        title: vocalsTitle || undefined,
        tags: vocalsTags || undefined,
      });
    } catch (error) {
      const message = error instanceof Error ? error.message : 'Не удалось добавить голос.';
      setFormError(message);
    }
  };

  const readySongs = songs.filter(s => Boolean(s.streamUrl ?? s.audioUrl));

  return (
    <div className="p-4">
      <h2 className="text-3xl font-bold text-white mb-6">Создать музыку</h2>
      
      {/* Tabs */}
      <div className="mb-6 flex gap-2 border-b border-white/10">
        <button
          onClick={() => setActiveTab('generate')}
          className={`px-4 py-2 font-semibold transition-colors ${
            activeTab === 'generate'
              ? 'text-purple-400 border-b-2 border-purple-400'
              : 'text-gray-400 hover:text-gray-300'
          }`}
        >
          Генерация
        </button>
        {onCover && (
          <button
            onClick={() => setActiveTab('cover')}
            className={`px-4 py-2 font-semibold transition-colors ${
              activeTab === 'cover'
                ? 'text-purple-400 border-b-2 border-purple-400'
                : 'text-gray-400 hover:text-gray-300'
            }`}
          >
            Cover
          </button>
        )}
        {onAddInstrumental && (
          <button
            onClick={() => setActiveTab('instrumental')}
            className={`px-4 py-2 font-semibold transition-colors ${
              activeTab === 'instrumental'
                ? 'text-purple-400 border-b-2 border-purple-400'
                : 'text-gray-400 hover:text-gray-300'
            }`}
          >
            Инструментал
          </button>
        )}
        {onAddVocals && (
          <button
            onClick={() => setActiveTab('vocals')}
            className={`px-4 py-2 font-semibold transition-colors ${
              activeTab === 'vocals'
                ? 'text-purple-400 border-b-2 border-purple-400'
                : 'text-gray-400 hover:text-gray-300'
            }`}
          >
            Голос
          </button>
        )}
      </div>
      {!isAuthenticated && (
        <div className="mb-6 rounded-xl border border-purple-500/30 bg-purple-500/10 px-4 py-3 text-sm text-purple-100">
          Войдите, чтобы сохранять и генерировать собственные треки. Нажмите «Сгенерировать»,
          и мы предложим авторизацию.
        </div>
      )}
      {formError && (
        <div className="mb-4 rounded-lg border border-red-500/40 bg-red-500/10 px-4 py-3 text-sm text-red-200">
          {formError}
        </div>
      )}
      {generationError && (
        <div className="mb-4 rounded-lg border border-red-500/40 bg-red-500/10 px-4 py-3 text-sm text-red-200">
          {generationError}
        </div>
      )}

      {/* Generate Tab */}
      {activeTab === 'generate' && (
        <>
          <fieldset disabled={isGenerating} className="space-y-6">
            <div>
              <label className="block text-lg font-semibold text-gray-300 mb-3">Жанр</label>
              <div className="flex flex-wrap gap-2">
                {genres.map((genre) => (
                  <Chip
                    key={genre}
                    label={genre}
                    isSelected={selectedGenre === genre}
                    onClick={() => setSelectedGenre(genre)}
                    disabled={isGenerating}
                  />
                ))}
              </div>
            </div>

            <div>
              <label htmlFor="prompt-input" className="block text-lg font-semibold text-gray-300 mb-3">
                Промпт (необязательно)
              </label>
              <textarea
                id="prompt-input"
                value={prompt}
                onChange={(e) => setPrompt(e.target.value)}
                placeholder="Например, синтвейв-трек для ночной поездки по киберпанк-городу"
                rows={3}
                className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
                disabled={isGenerating}
              />
            </div>
          </fieldset>

          <div className="mt-8">
            <button
              onClick={handleGenerate}
              disabled={isGenerating}
              className="w-full bg-gradient-to-r from-purple-600 to-indigo-600 text-white font-bold py-4 px-4 rounded-full flex items-center justify-center text-lg hover:opacity-90 transition-all disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isGenerating ? (
                <>
                  <SpinnerIcon className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" />
                  Генерация...
                </>
              ) : (
                'Сгенерировать'
              )}
            </button>
          </div>
        </>
      )}

      {/* Cover Tab */}
      {activeTab === 'cover' && onCover && (
        <fieldset disabled={isGenerating} className="space-y-6">
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Аудио файл *</label>
            <input
              type="file"
              accept="audio/mp3,audio/wav,audio/m4a"
              onChange={(e) => setCoverFile(e.target.files?.[0] || null)}
              className="w-full bg-[#30313d] text-white rounded-lg p-3 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-purple-500 file:text-white hover:file:bg-purple-400"
              disabled={isGenerating}
            />
            {coverFile && (
              <p className="mt-2 text-sm text-gray-400">Выбран: {coverFile.name}</p>
            )}
          </div>
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Промпт (необязательно)</label>
            <textarea
              value={coverPrompt}
              onChange={(e) => setCoverPrompt(e.target.value)}
              placeholder="Опишите желаемый стиль cover"
              rows={3}
              className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            />
          </div>
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Название (необязательно)</label>
            <input
              type="text"
              value={coverTitle}
              onChange={(e) => setCoverTitle(e.target.value)}
              placeholder="Название трека"
              className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            />
          </div>
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Теги (необязательно)</label>
            <input
              type="text"
              value={coverTags}
              onChange={(e) => setCoverTags(e.target.value)}
              placeholder="Теги через запятую"
              className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            />
          </div>
          <button
            onClick={handleCover}
            disabled={isGenerating || !coverFile}
            className="w-full bg-gradient-to-r from-purple-600 to-indigo-600 text-white font-bold py-4 px-4 rounded-full flex items-center justify-center text-lg hover:opacity-90 transition-all disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {isGenerating ? (
              <>
                <SpinnerIcon className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" />
                Генерация cover...
              </>
            ) : (
              'Сгенерировать cover'
            )}
          </button>
        </fieldset>
      )}

      {/* Instrumental Tab */}
      {activeTab === 'instrumental' && onAddInstrumental && (
        <fieldset disabled={isGenerating} className="space-y-6">
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Выберите трек *</label>
            <select
              value={selectedSongForInstrumental?.id || ''}
              onChange={(e) => {
                const song = readySongs.find(s => s.id === e.target.value);
                setSelectedSongForInstrumental(song || null);
              }}
              className="w-full bg-[#30313d] text-white rounded-lg p-3 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            >
              <option value="">-- Выберите трек --</option>
              {readySongs.map(song => (
                <option key={song.id} value={song.id}>{song.title}</option>
              ))}
            </select>
          </div>
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Промпт (необязательно)</label>
            <textarea
              value={instrumentalPrompt}
              onChange={(e) => setInstrumentalPrompt(e.target.value)}
              placeholder="Опишите желаемый инструментал"
              rows={3}
              className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            />
          </div>
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Стиль (необязательно)</label>
            <input
              type="text"
              value={instrumentalStyle}
              onChange={(e) => setInstrumentalStyle(e.target.value)}
              placeholder="Стиль инструментала"
              className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            />
          </div>
          <button
            onClick={handleAddInstrumental}
            disabled={isGenerating || !selectedSongForInstrumental}
            className="w-full bg-gradient-to-r from-purple-600 to-indigo-600 text-white font-bold py-4 px-4 rounded-full flex items-center justify-center text-lg hover:opacity-90 transition-all disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {isGenerating ? (
              <>
                <SpinnerIcon className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" />
                Добавление инструментала...
              </>
            ) : (
              'Добавить инструментал'
            )}
          </button>
        </fieldset>
      )}

      {/* Vocals Tab */}
      {activeTab === 'vocals' && onAddVocals && (
        <fieldset disabled={isGenerating} className="space-y-6">
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Выберите трек *</label>
            <select
              value={selectedSongForVocals?.id || ''}
              onChange={(e) => {
                const song = readySongs.find(s => s.id === e.target.value);
                setSelectedSongForVocals(song || null);
              }}
              className="w-full bg-[#30313d] text-white rounded-lg p-3 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            >
              <option value="">-- Выберите трек --</option>
              {readySongs.map(song => (
                <option key={song.id} value={song.id}>{song.title}</option>
              ))}
            </select>
          </div>
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Промпт (необязательно)</label>
            <textarea
              value={vocalsPrompt}
              onChange={(e) => setVocalsPrompt(e.target.value)}
              placeholder="Опишите желаемый вокал"
              rows={3}
              className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            />
          </div>
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Стиль (необязательно)</label>
            <input
              type="text"
              value={vocalsStyle}
              onChange={(e) => setVocalsStyle(e.target.value)}
              placeholder="Стиль вокала"
              className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            />
          </div>
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Текст песни (необязательно)</label>
            <textarea
              value={vocalsLyrics}
              onChange={(e) => setVocalsLyrics(e.target.value)}
              placeholder="Текст песни"
              rows={5}
              className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            />
          </div>
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Название (необязательно)</label>
            <input
              type="text"
              value={vocalsTitle}
              onChange={(e) => setVocalsTitle(e.target.value)}
              placeholder="Название трека"
              className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            />
          </div>
          <div>
            <label className="block text-lg font-semibold text-gray-300 mb-3">Теги (необязательно)</label>
            <input
              type="text"
              value={vocalsTags}
              onChange={(e) => setVocalsTags(e.target.value)}
              placeholder="Теги через запятую"
              className="w-full bg-[#30313d] text-white rounded-lg p-3 placeholder-gray-500 focus:ring-2 focus:ring-purple-500 focus:outline-none transition"
              disabled={isGenerating}
            />
          </div>
          <button
            onClick={handleAddVocals}
            disabled={isGenerating || !selectedSongForVocals}
            className="w-full bg-gradient-to-r from-purple-600 to-indigo-600 text-white font-bold py-4 px-4 rounded-full flex items-center justify-center text-lg hover:opacity-90 transition-all disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {isGenerating ? (
              <>
                <SpinnerIcon className="animate-spin -ml-1 mr-3 h-5 w-5 text-white" />
                Добавление голоса...
              </>
            ) : (
              'Добавить голос'
            )}
          </button>
        </fieldset>
      )}

      {latestTask && (
        <div className="mt-10 space-y-6">
          <h3 className="text-xl font-semibold text-white">Последняя генерация</h3>

          <div className="space-y-4">
            {latestTask.songs.map((song) => {
              const artwork = song.artworkUrl ?? '/images/generated.jpg';
              const isReady = Boolean(song.streamUrl ?? song.audioUrl);
              const audioUrl = song.streamUrl ?? song.audioUrl ?? song.sourceStreamUrl ?? song.sourceAudioUrl;
              
              return (
                <div
                  key={song.id}
                  className="flex flex-col gap-4 rounded-2xl border border-white/5 bg-[#262633] p-4 md:flex-row md:items-center"
                >
                  <div className="flex items-center gap-4 flex-1">
                    <div className="relative h-20 w-20 flex-shrink-0">
                      <img
                        src={artwork}
                        alt={song.title}
                        className="h-20 w-20 rounded-xl object-cover"
                      />
                      {!isReady && (
                        <div className="absolute inset-0 flex items-center justify-center bg-black/50 rounded-xl">
                          <SpinnerIcon className="h-6 w-6 animate-spin text-purple-400" />
                        </div>
                      )}
                    </div>
                    <div className="min-w-0 flex-1">
                      <p className="text-lg font-semibold text-white truncate">{song.title}</p>
                      <p className="text-sm text-gray-400 truncate">{song.artist ?? 'Suno AI'}{song.model ? ` • ${song.model}` : ''}</p>
                      {song.tags && <p className="text-xs text-gray-500 truncate mt-1">Теги: {song.tags}</p>}
                      {song.prompt && <p className="text-xs text-gray-500 mt-1 overflow-hidden text-ellipsis line-clamp-2">Промпт: {song.prompt}</p>}
                    </div>
                  </div>
                  <div className="w-full md:w-auto">
                    {isReady && audioUrl ? (
                      <div className="space-y-3">
                        <audio
                          ref={(el) => {
                            if (el) {
                              audioRefs.current.set(song.id, el);
                            } else {
                              audioRefs.current.delete(song.id);
                            }
                          }}
                          controls
                          autoPlay
                          src={audioUrl}
                          className="w-full"
                          onLoadedData={(e) => {
                            // Автоматически запускаем воспроизведение когда трек готов
                            const audio = e.currentTarget;
                            if (audio.paused) {
                              audio.play().catch((err) => {
                                console.log('Auto-play prevented:', err);
                              });
                            }
                          }}
                          onCanPlay={(e) => {
                            // Также пытаемся запустить при canPlay
                            const audio = e.currentTarget;
                            if (audio.paused && !audio.ended) {
                              audio.play().catch((err) => {
                                console.log('Auto-play on canPlay prevented:', err);
                              });
                            }
                          }}
                        />
                        <button
                          type="button"
                          onClick={() => void handleExtend(song)}
                          disabled={isGenerating || extendingSongId === song.id}
                          className="flex w-full items-center justify-center gap-2 rounded-full border border-purple-500/40 px-4 py-2 text-sm font-semibold text-purple-200 transition hover:border-purple-400 hover:text-white disabled:cursor-not-allowed disabled:opacity-60"
                        >
                          {extendingSongId === song.id ? (
                            <>
                              <SpinnerIcon className="h-4 w-4 animate-spin" />
                              Продлеваем...
                            </>
                          ) : (
                            'Продлить трек'
                          )}
                        </button>
                      </div>
                    ) : (
                      <div className="flex flex-col items-center justify-center gap-3 p-6 rounded-xl bg-[#1a1b24] border border-white/5">
                        <div className="relative">
                          <div className="h-16 w-16 rounded-full border-4 border-purple-500/20"></div>
                          <div className="absolute inset-0 h-16 w-16 rounded-full border-4 border-transparent border-t-purple-500 animate-spin"></div>
                        </div>
                        <div className="text-center">
                          <p className="text-sm font-semibold text-white">Готовим трек...</p>
                          <p className="text-xs text-gray-400 mt-1">Ждём аудио</p>
                        </div>
                      </div>
                    )}
                  </div>
                </div>
              );
            })}
          </div>
        </div>
      )}
    </div>
  );
};

export default CreatePage;