import React, { useState, useEffect } from 'react';
import { Song, GenerationTask } from '../types';
import SongItem from './SongItem';
import { ChevronLeft, ChevronRight, Grid3x3 } from 'lucide-react';

interface DesktopHomePageProps {
  songs: Song[];
  publicSongs?: Song[];
  playingSongId: string | null;
  onPlaySong: (song: Song) => void;
  onLoadPublicSongs?: (filter: string) => Promise<void>;
  onPublishSong?: (song: Song) => Promise<void>;
  onUnpublishSong?: (song: Song) => Promise<void>;
  isAuthenticated?: boolean;
  onGenerate?: (payload: {
    prompt: string;
    genre?: string | null;
    mood?: string | null;
    duration?: number;
    customMode?: boolean;
    instrumental?: boolean;
    model?: 'V3_5' | 'V4' | 'V4_5' | 'V4_5PLUS' | 'V5';
  }) => Promise<void>;
  isGenerating?: boolean;
  latestTask?: GenerationTask | null;
  generationError?: string | null;
  onExtendSong?: (song: Song) => Promise<void>;
  extendingSongId?: string | null;
  onRequireAuth?: () => void;
}

const DesktopHomePage: React.FC<DesktopHomePageProps> = ({
  songs,
  publicSongs = [],
  playingSongId,
  onPlaySong,
  onLoadPublicSongs,
  onPublishSong,
  onUnpublishSong,
  isAuthenticated = false,
  onGenerate,
  isGenerating = false,
  latestTask = null,
  generationError = null,
  onExtendSong,
  extendingSongId = null,
  onRequireAuth,
}) => {
  const [activeSlide, setActiveSlide] = useState(0);
  const [activeTab, setActiveTab] = useState<'songs' | 'playlists'>('songs');
  const [activeFilter, setActiveFilter] = useState('Выбор редакции');
  const slides = 5;

  const handleTogglePlay = (songId: string) => {
    const songToPlay = displaySongs.find((s) => s.id === songId);
    if (songToPlay) {
      onPlaySong(songToPlay);
    }
  };

  const filters = ['Выбор редакции', 'Сегодня', 'Неделя', 'Месяц', 'Все', 'В эфире'];

  // Load public songs when filter changes to "В эфире"
  useEffect(() => {
    if (activeFilter === 'В эфире' && onLoadPublicSongs) {
      onLoadPublicSongs(activeFilter);
    }
  }, [activeFilter, onLoadPublicSongs]);


  // Determine which songs to display based on active filter
  const displaySongs = activeFilter === 'В эфире' ? publicSongs : songs;

  const formatDuration = (seconds: number | null | undefined): string => {
    if (!seconds) return '0:00';
    const mins = Math.floor(seconds / 60);
    const secs = Math.floor(seconds % 60);
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  return (
    <div className="flex-1 overflow-y-auto">
      <div className="mx-auto max-w-[1920px] px-8 py-6">
        {/* Featured Banner */}
        <section className="mb-8">
          <div className="relative h-64 overflow-hidden rounded-2xl">
            <div
              className="h-full w-full bg-cover bg-center"
              style={{
                backgroundImage: "url('/images/banner.jpg')",
              }}
            >
              <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/40 to-transparent" />
            </div>
            <div className="absolute inset-0 flex items-end p-8">
              <div className="relative z-10">
                <p className="text-xs font-semibold uppercase tracking-widest text-gray-300">
                  ВСТРЕЧАЙТЕ ПРОДЮСЕРА
                </p>
                <h2 className="mt-2 text-4xl font-bold text-white">
                  Первый ИИ-агент для музыки
                </h2>
              </div>
            </div>
            {/* Navigation Arrows */}
              <button
                onClick={() => setActiveSlide((prev) => (prev > 0 ? prev - 1 : slides - 1))}
                className="absolute left-4 top-1/2 -translate-y-1/2 rounded-full bg-black/50 p-2 text-white transition hover:bg-black/70"
                aria-label="Предыдущий слайд"
              >
                <ChevronLeft className="h-5 w-5" />
              </button>
              <button
                onClick={() => setActiveSlide((prev) => (prev < slides - 1 ? prev + 1 : 0))}
                className="absolute right-4 top-1/2 -translate-y-1/2 rounded-full bg-black/50 p-2 text-white transition hover:bg-black/70"
                aria-label="Следующий слайд"
              >
                <ChevronRight className="h-5 w-5" />
              </button>
            {/* Progress Indicator */}
            <div className="absolute right-8 top-4 flex gap-1">
              {Array.from({ length: slides }).map((_, index) => (
                <div
                  key={index}
                  className={`h-1 w-8 rounded-full transition-colors ${
                    index === activeSlide ? 'bg-white' : 'bg-white/30'
                  }`}
                />
              ))}
            </div>
          </div>
        </section>

        {/* Songs/Playlists Tabs */}
        <section>
          <div className="mb-6 flex items-center justify-between border-b border-white/10">
            <div className="flex gap-8">
              <button
                onClick={() => setActiveTab('songs')}
                className={`border-b-2 pb-3 text-sm font-semibold transition ${
                  activeTab === 'songs'
                    ? 'border-white text-white'
                    : 'border-transparent text-gray-400 hover:text-white'
                }`}
              >
                Треки
              </button>
              <button
                onClick={() => setActiveTab('playlists')}
                className={`border-b-2 pb-3 text-sm font-semibold transition ${
                  activeTab === 'playlists'
                    ? 'border-white text-white'
                    : 'border-transparent text-gray-400 hover:text-white'
                }`}
              >
                Плейлисты
              </button>
            </div>
            {activeTab === 'songs' && (
              <button
                className="text-gray-400 transition hover:text-white"
                aria-label="Переключить вид сетки"
              >
                <Grid3x3 className="h-5 w-5" />
              </button>
            )}
          </div>

          {activeTab === 'songs' && (
            <>
              {/* Filter Buttons */}
              <div className="mb-6 flex items-center gap-2">
                {filters.map((filter) => (
                  <button
                    key={filter}
                    onClick={() => setActiveFilter(filter)}
                    className={`rounded-full px-4 py-2 text-sm font-semibold transition ${
                      activeFilter === filter
                        ? 'bg-purple-500/20 text-white'
                        : 'bg-white/5 text-gray-400 hover:bg-white/10 hover:text-white'
                    }`}
                  >
                    {filter}
                  </button>
                ))}
              </div>

              {/* Songs List */}
              {displaySongs.length === 0 ? (
                <div className="py-12 text-center">
                  <p className="text-gray-400">
                    {activeFilter === 'В эфире'
                      ? 'Здесь появятся публичные треки после публикации.'
                      : 'Здесь появятся ваши сгенерированные треки после завершения генерации.'}
                  </p>
                </div>
              ) : (
                <div className="space-y-0">
                  {displaySongs.map((song, index) => (
                    <div
                      key={song.id}
                      className="group flex items-center gap-4 rounded-lg px-4 py-3 transition hover:bg-white/5"
                    >
                      <span className="w-8 text-sm text-gray-400">
                        {String(index + 1).padStart(2, '0')}
                      </span>
                      <img
                        src={song.artworkUrl ?? '/images/generated.jpg'}
                        alt={song.title}
                        className="h-12 w-12 flex-shrink-0 rounded object-cover"
                      />
                      <div className="min-w-0 flex-1">
                        <p className="truncate font-semibold text-white">{song.title}</p>
                        <p className="truncate text-sm text-gray-400">
                          {song.artist ?? 'Suno AI'}
                        </p>
                      </div>
                      <div className="flex items-center gap-6 text-sm text-gray-400">
                        <div className="flex items-center gap-1">
                          <svg className="h-4 w-4" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M8 5v14l11-7z" />
                          </svg>
                          <span>{song.playsCount ? (song.playsCount / 1000).toFixed(1) + 'k' : '0'}</span>
                        </div>
                        <div className="flex items-center gap-1">
                          <svg className="h-4 w-4" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M12 21.35l-1.45-1.32C5.4 15.36 2 12.28 2 8.5 2 5.42 4.42 3 7.5 3c1.74 0 3.41.81 4.5 2.09C13.09 3.81 14.76 3 16.5 3 19.58 3 22 5.42 22 8.5c0 3.78-3.4 6.86-8.55 11.54L12 21.35z" />
                          </svg>
                          <span>{song.likesCount ?? 0}</span>
                        </div>
                        <span>{formatDuration(song.durationSeconds)}</span>
                      </div>
                      <button
                        onClick={() => handleTogglePlay(song.id)}
                        className="opacity-0 transition-opacity group-hover:opacity-100"
                        aria-label={playingSongId === song.id ? 'Пауза' : 'Воспроизвести'}
                      >
                        {playingSongId === song.id ? (
                          <svg className="h-5 w-5 text-gray-400" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M6 4h4v16H6V4zm8 0h4v16h-4V4z" />
                          </svg>
                        ) : (
                          <svg className="h-5 w-5 text-gray-400" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M8 5v14l11-7z" />
                          </svg>
                        )}
                      </button>
                    </div>
                  ))}
                </div>
              )}
            </>
          )}

          {activeTab === 'playlists' && (
            <div className="py-12 text-center">
              <p className="text-gray-400">Плейлисты скоро появятся</p>
            </div>
          )}
        </section>
      </div>

    </div>
  );
};

export default DesktopHomePage;
