import React from 'react';
import {
  HomeIcon,
  LibraryIcon,
  GenerationsIcon,
  UserIcon,
  HeartIcon,
  Flame,
  Folder,
  ListMusic,
  MoreHorizontalIcon,
} from './Icons';

interface DesktopSidebarProps {
  activeTab: string;
  setActiveTab: (tab: string) => void;
  isAuthenticated: boolean;
  onAuthClick?: () => void;
  onCreateClick?: () => void;
}

const DesktopSidebar: React.FC<DesktopSidebarProps> = ({
  activeTab,
  setActiveTab,
  isAuthenticated,
  onAuthClick,
  onCreateClick,
}) => {
  const exploreItems = [
    { id: 'home', label: 'Рекомендуемое', icon: GenerationsIcon },
    { id: 'following', label: 'Подписки', icon: UserIcon },
    { id: 'personalize', label: 'Персонализация', icon: HeartIcon },
  ];

  const libraryItems = [
    { id: 'library', label: 'Треки', icon: LibraryIcon },
    { id: 'vibes', label: 'Настроения', icon: Flame },
    { id: 'projects', label: 'Проекты', icon: Folder },
    { id: 'playlists', label: 'Плейлисты', icon: ListMusic },
  ];

  const handleItemClick = (id: string) => {
    if (id === 'more' && !isAuthenticated) {
      onAuthClick?.();
      return;
    }
    setActiveTab(id);
  };

  return (
    <aside className="fixed left-0 top-[73px] z-40 h-[calc(100vh-73px)] w-64 border-r border-white/10 bg-[#1a1b24]/95 backdrop-blur">
      <div className="flex h-full flex-col px-4 py-6">
        {/* Explore Section */}
        <div className="mb-8">
          <h2 className="mb-3 px-3 text-xs font-semibold uppercase tracking-wider text-gray-500">
            Обзор
          </h2>
          <nav className="space-y-1">
            {exploreItems.map((item) => {
              const Icon = item.icon;
              const isActive = activeTab === item.id;
              return (
                <button
                  key={item.id}
                  onClick={() => handleItemClick(item.id)}
                  className={`flex w-full items-center gap-3 rounded-lg px-3 py-2.5 text-sm font-medium transition ${
                    isActive
                      ? 'bg-purple-500/20 text-white'
                      : 'text-gray-400 hover:bg-white/5 hover:text-white'
                  }`}
                >
                  <Icon className="h-5 w-5" />
                  {item.label}
                </button>
              );
            })}
          </nav>
        </div>

        {/* Library Section */}
        <div className="mb-auto">
          <h2 className="mb-3 px-3 text-xs font-semibold uppercase tracking-wider text-gray-500">
            Библиотека
          </h2>
          <nav className="space-y-1">
            {libraryItems.map((item) => {
              const Icon = item.icon;
              const isActive = activeTab === item.id;
              return (
                <button
                  key={item.id}
                  onClick={() => handleItemClick(item.id)}
                  className={`flex w-full items-center gap-3 rounded-lg px-3 py-2.5 text-sm font-medium transition ${
                    isActive
                      ? 'bg-purple-500/20 text-white'
                      : 'text-gray-400 hover:bg-white/5 hover:text-white'
                  }`}
                >
                  <Icon className="h-5 w-5" />
                  {item.label}
                </button>
              );
            })}
          </nav>
        </div>

        {/* Generate Button */}
        <button
          onClick={() => {
            if (!isAuthenticated) {
              onAuthClick?.();
            } else {
              onCreateClick?.();
            }
          }}
          className="mb-3 w-full rounded-xl bg-gradient-to-r from-purple-600 to-indigo-600 px-4 py-3.5 text-sm font-bold text-white shadow-lg shadow-purple-500/30 transition-all hover:scale-[1.02] hover:shadow-xl hover:shadow-purple-500/40 active:scale-[0.98]"
        >
          Генерировать
        </button>

        {/* More Button */}
        <button
          onClick={() => handleItemClick('more')}
          className={`flex w-full items-center gap-3 rounded-lg px-3 py-2.5 text-sm font-medium transition ${
            activeTab === 'more'
              ? 'bg-purple-500/20 text-white'
              : 'text-gray-400 hover:bg-white/5 hover:text-white'
          }`}
        >
          <MoreHorizontalIcon className="h-5 w-5" />
          Еще
        </button>
      </div>
    </aside>
  );
};

export default DesktopSidebar;

