import React from 'react';
import { AuthUser } from '../types';
import { LogoIcon, SearchIcon } from './Icons';

interface HeaderProps {
  user: AuthUser | null;
  onLogin?: () => void;
  onLogout?: () => void;
  isAuthenticating?: boolean;
  showLogout?: boolean;
}

const Header: React.FC<HeaderProps> = ({
  user,
  onLogin,
  onLogout,
  isAuthenticating,
  showLogout = true,
}) => {
  const displayName =
    user?.name ??
    ([user?.telegram_first_name, user?.telegram_last_name].filter(Boolean).join(' ') ||
      (user?.telegram_username ? `@${user.telegram_username}` : 'Гость'));

  return (
    <header className="sticky top-0 z-10 border-b border-white/10 bg-[#1a1b24]/95 p-4 backdrop-blur">
      <div className="flex items-center justify-between gap-4">
        <div className="flex items-center gap-2">
          <LogoIcon />
          <h1 className="text-2xl font-bold tracking-tighter text-white">Melodist</h1>
        </div>

        <div className="flex items-center gap-3">
          <SearchIcon className="hidden h-6 w-6 text-gray-400 sm:block" />
          {user ? (
            <div className="hidden text-right sm:block">
              <p className="text-sm font-semibold text-white">{displayName}</p>
              {user.is_telegram && user.telegram_username && (
                <p className="text-xs text-gray-400">@{user.telegram_username}</p>
              )}
            </div>
          ) : (
            <p className="hidden text-sm text-gray-400 sm:block">Гость</p>
          )}
          {user ? (
            onLogout &&
            showLogout && (
              <button
                type="button"
                onClick={() => {
                  onLogout();
                }}
                className="rounded-full border border-white/10 px-3 py-1 text-xs font-semibold text-gray-300 transition hover:border-purple-400 hover:text-white"
              >
                Выйти
              </button>
            )
          ) : (
            onLogin && (
              <button
                type="button"
                onClick={() => {
                  onLogin();
                }}
                disabled={isAuthenticating}
                className="rounded-full border border-purple-400/60 px-3 py-1 text-xs font-semibold text-purple-300 transition hover:border-purple-400 hover:text-white disabled:cursor-not-allowed disabled:opacity-60"
              >
                {isAuthenticating ? 'Подключаем...' : 'Войти'}
              </button>
            )
          )}
        </div>
      </div>
    </header>
  );
};

export default Header;
