import React from 'react';
import { Song } from '../types';
import { LibraryIcon, PlayIcon } from './Icons';

interface LibraryPageProps {
  songs: Song[];
  onPlaySong: (song: Song) => void;
  currentSongId: string | null;
  isPlaying: boolean;
  onPublishSong?: (song: Song) => Promise<void>;
  onUnpublishSong?: (song: Song) => Promise<void>;
  isAuthenticated?: boolean;
}

const LibraryPage: React.FC<LibraryPageProps> = ({
  songs,
  onPlaySong,
  currentSongId,
  isPlaying,
  onPublishSong,
  onUnpublishSong,
  isAuthenticated = false,
}) => {
  if (songs.length === 0) {
    return (
      <div className="p-6 flex flex-col items-center justify-center text-center h-full pt-20 animate-fade-in">
        <LibraryIcon className="w-16 h-16 text-gray-500 mb-4" />
        <h2 className="text-2xl font-bold text-white">Здесь появятся ваши треки</h2>
        <p className="text-gray-400 mt-2">
          Сгенерируйте музыку — и она автоматически попадёт в библиотеку.
        </p>
      </div>
    );
  }

  return (
    <div className="p-4 space-y-6 animate-fade-in">
      <div>
        <h2 className="text-3xl font-bold text-white">Мои треки</h2>
        <p className="text-sm text-gray-400 mt-1">
          Все композиции, которые вы сгенерировали. Когда трек будет готов, его можно послушать.
        </p>
      </div>

      <div className="grid gap-4 md:grid-cols-2">
        {songs.map((song) => {
          const artwork = song.artworkUrl ?? '/images/generated.jpg';
          const playingNow = currentSongId === song.id;
          const isReady = Boolean(
            song.streamUrl ??
              song.audioUrl ??
              song.sourceStreamUrl ??
              song.sourceAudioUrl,
          );
          const statusLabel = song.isPlaceholder
            ? 'Готовим трек...'
            : isReady
              ? 'Готов к прослушиванию'
              : 'Идёт обработка';

          return (
            <div
              key={song.id}
              className={`flex items-center gap-4 rounded-2xl border border-white/5 bg-[#1f2030] p-4 transition-all hover:border-purple-400/50 ${
                playingNow ? 'border-purple-500/60 shadow-lg shadow-purple-500/20' : ''
              }`}
            >
              <img
                src={artwork}
                alt={song.title}
                className="h-20 w-20 flex-shrink-0 rounded-xl object-cover"
              />
              <div className="flex-1 min-w-0">
                <p className="text-lg font-semibold text-white truncate">{song.title}</p>
                <p className="text-sm text-gray-400 truncate">
                  {song.artist ?? 'Suno AI'}
                </p>
                <p className="text-xs text-gray-500 mt-1">{statusLabel}</p>
                {song.tags && (
                  <p className="text-xs text-gray-500 truncate mt-1">Теги: {song.tags}</p>
                )}
                <div className="mt-3 flex flex-wrap gap-2">
                  <button
                    type="button"
                    onClick={() => onPlaySong(song)}
                    disabled={!isReady}
                    className="flex items-center gap-2 rounded-full bg-purple-500 px-4 py-2 text-sm font-semibold text-white transition hover:bg-purple-400 disabled:cursor-not-allowed disabled:opacity-60"
                  >
                    <PlayIcon className="h-4 w-4" />
                    {playingNow && isPlaying
                      ? 'Сейчас играет'
                      : isReady
                        ? 'Слушать'
                        : 'Ждём аудио'}
                  </button>
                  {isAuthenticated && isReady && (
                    <button
                      type="button"
                      onClick={async () => {
                        if (song.isPublic && onUnpublishSong) {
                          try {
                            await onUnpublishSong(song);
                          } catch (error) {
                            console.error('Failed to unpublish:', error);
                          }
                        } else if (!song.isPublic && onPublishSong) {
                          try {
                            await onPublishSong(song);
                          } catch (error) {
                            console.error('Failed to publish:', error);
                          }
                        }
                      }}
                      className={`flex items-center gap-2 rounded-full px-4 py-2 text-sm font-semibold transition ${
                        song.isPublic
                          ? 'bg-green-500/20 text-green-400 hover:bg-green-500/30'
                          : 'bg-white/10 text-gray-300 hover:bg-white/20'
                      }`}
                    >
                      {song.isPublic ? '✓ Опубликован' : '📤 Опубликовать'}
                    </button>
                  )}
                </div>
              </div>
            </div>
          );
        })}
      </div>
    </div>
  );
};

export default LibraryPage;
