import React from 'react';
import { AuthUser } from '../types';
import { BalanceIcon, GenerationsIcon } from './Icons';

interface MorePageProps {
  isAuthenticated: boolean;
  user: AuthUser | null;
  onLogin?: () => void;
  onLogout?: () => void;
  isTelegramContext: boolean;
  generationCount: number;
  tokenBalance: number;
}

const MorePage: React.FC<MorePageProps> = ({
  isAuthenticated,
  user,
  onLogin,
  onLogout,
  isTelegramContext,
  generationCount,
  tokenBalance,
}) => {
  if (!isAuthenticated || !user) {
    return (
      <div className="flex min-h-[calc(100vh-8rem)] flex-col items-center justify-center gap-6 px-6 text-center animate-fade-in">
        <div className="space-y-3">
          <h1 className="text-3xl font-bold text-white">Войдите в Melodist</h1>
          <p className="text-sm text-gray-400">
            Авторизуйтесь через Telegram, чтобы сохранять прогресс, отслеживать генерации и получать бонусные токены.
          </p>
        </div>
        <button
          type="button"
          onClick={onLogin}
          className="w-full max-w-xs rounded-full bg-purple-500 px-6 py-3 text-sm font-semibold text-white shadow-lg shadow-purple-500/30 transition hover:bg-purple-400"
        >
          Войти через Telegram
        </button>
      </div>
    );
  }

  const displayName =
    user.name ??
    ([user.telegram_first_name, user.telegram_last_name].filter(Boolean).join(' ') ||
      (user.telegram_username ? `@${user.telegram_username}` : 'Пользователь'));

  const avatarUrl = user.telegram_photo_url ?? '/images/avatar1.jpg';
  const usernameLine = user.telegram_username ? `@${user.telegram_username}` : null;
  const safeBalance = Number.isFinite(tokenBalance) ? tokenBalance : 0;
  const safeGenerations = Number.isFinite(generationCount) ? generationCount : 0;

  return (
    <div className="space-y-8 px-6 py-8 animate-fade-in">
      <div className="flex items-center gap-4 rounded-2xl border border-white/10 bg-[#1f2030] p-6">
        <img
          src={avatarUrl}
          alt={displayName}
          className="h-20 w-20 flex-shrink-0 rounded-2xl object-cover shadow-lg shadow-black/40"
        />
        <div className="min-w-0">
          <h2 className="text-xl font-semibold text-white truncate">{displayName}</h2>
          {usernameLine && <p className="text-sm text-gray-400">{usernameLine}</p>}
          {user.last_login_at && (
            <p className="mt-1 text-xs text-gray-500">
              Последний вход: {new Date(user.last_login_at).toLocaleString('ru-RU')}
            </p>
          )}
        </div>
      </div>

      <div className="grid gap-4 md:grid-cols-2">
        <div className="rounded-2xl border border-purple-500/20 bg-[#1f2030] p-5 shadow-lg shadow-purple-500/10">
          <div className="flex items-center justify-between">
            <span className="text-sm uppercase tracking-widest text-gray-400">Генераций</span>
            <GenerationsIcon className="h-5 w-5 text-purple-300" />
          </div>
          <p className="mt-3 text-3xl font-semibold text-white">{safeGenerations}</p>
          <p className="mt-1 text-xs text-gray-500">Всего созданных треков с начала использования</p>
        </div>

        <div className="rounded-2xl border border-emerald-500/20 bg-[#1f2030] p-5 shadow-lg shadow-emerald-500/10">
          <div className="flex items-center justify-between">
            <span className="text-sm uppercase tracking-widest text-gray-400">Баланс токенов</span>
            <BalanceIcon className="h-5 w-5 text-emerald-300" />
          </div>
          <p className="mt-3 text-3xl font-semibold text-white">{safeBalance}</p>
          <p className="mt-1 text-xs text-gray-500">Токены можно обменять на новые генерации</p>
        </div>
      </div>

      {!isTelegramContext && onLogout && (
        <button
          type="button"
          onClick={onLogout}
          className="w-full rounded-full border border-white/10 bg-black/30 px-6 py-3 text-sm font-semibold text-gray-200 transition hover:border-red-400 hover:text-white"
        >
          Выйти из аккаунта
        </button>
      )}
    </div>
  );
};

export default MorePage;