import React from 'react';
import { Song } from '../types';
import { MoreHorizontalIcon, SoundWaveIcon } from './Icons';

interface SongItemProps {
  song: Song;
  isPlaying: boolean;
  onTogglePlay: (id: string) => void;
  rank?: string;
}

const SongItem: React.FC<SongItemProps> = ({ song, isPlaying, onTogglePlay, rank }) => {
  const artwork = song.artworkUrl ?? '/images/generated.jpg';
  const secondaryLine = song.artist ?? 'Suno AI';
  const isReady = Boolean(
    song.streamUrl ??
      song.audioUrl ??
      song.sourceStreamUrl ??
      song.sourceAudioUrl,
  );
  const statusLabel = song.isPlaceholder
    ? 'Готовим трек...'
    : !isReady
      ? 'Идёт обработка'
      : song.status && song.status !== 'complete'
        ? `Статус: ${song.status}`
        : null;
  const subtitleParts = [
    secondaryLine,
    song.model ? `Модель ${song.model}` : null,
    statusLabel,
  ].filter(Boolean);

  const handleClick = () => {
    if (!isReady) {
      return;
    }
    onTogglePlay(song.id);
  };

  return (
    <div className="flex items-center space-x-4 p-2 rounded-lg hover:bg-[#30313d]/60 transition-colors">
      {rank && <div className="text-gray-400 font-bold text-lg w-6 text-center">{rank}</div>}
      <div className="relative flex-shrink-0">
          <img src={artwork} alt={song.title} className="w-16 h-16 rounded-lg object-cover" />
          <button
            type="button"
            onClick={handleClick}
            disabled={!isReady}
            className="absolute inset-0 bg-black/30 flex items-center justify-center rounded-lg opacity-0 hover:opacity-100 transition-opacity disabled:opacity-60 disabled:cursor-not-allowed"
            aria-label={`Play ${song.title}`}
          >
            {isPlaying && <SoundWaveIcon className="text-white" />}
          </button>
          {isPlaying && (
              <div className="absolute inset-0 bg-black/50 flex items-center justify-center rounded-lg">
                  <SoundWaveIcon className="text-white" />
              </div>
          )}
      </div>
      <div
        className="flex-1 min-w-0"
        onClick={handleClick}
        role="button"
        tabIndex={0}
        onKeyDown={(event) => {
          if ((event.key === 'Enter' || event.key === ' ') && isReady) {
            event.preventDefault();
            handleClick();
          }
        }}
        aria-disabled={!isReady}
      >
        <h3 className="text-white font-semibold truncate">{song.title}</h3>
        <p className="text-gray-400 text-sm truncate mt-1">
          {subtitleParts.join(' • ')}
        </p>
      </div>
      <button className="text-gray-400 hover:text-white">
        <MoreHorizontalIcon className="w-6 h-6" />
      </button>
    </div>
  );
};

export default SongItem;